"use server";

import { AmplifyServer } from '@aws-amplify/core/internals/adapter-core';
import { fetchAuthSession, getCurrentUser } from "aws-amplify/auth/server";
import { cookies } from "next/headers";
import { runWithAmplifyServerContext } from "./context";
import { AuthSession, signOut } from 'aws-amplify/auth';
import { AppUserGroup } from '@/zod/amplify.schema';

export const getCurrentSessionFromServer = async () => {
  const currentSession = await runWithAmplifyServerContext({
    nextServerContext: { cookies },
    operation: (contextSpec: AmplifyServer.ContextSpec) => fetchAuthSession(contextSpec),
  });
  return currentSession;
};

export const getCurrentUserFromServer = async () => {
  const currentUser = runWithAmplifyServerContext({
    nextServerContext: { cookies },
    operation: async (contextSpec: AmplifyServer.ContextSpec) => {
      const user = await getCurrentUser(contextSpec);
      return {
        name: user.signInDetails?.loginId ?? 'N/A'
      }
    },
  });
  return currentUser;
};

export const getIsUserAuthenticatedFromServer = async () => {
  const isAuthenticated = await runWithAmplifyServerContext({
    nextServerContext: { cookies },
    operation: async (contextSpec: AmplifyServer.ContextSpec) => {
      try {
        const session = await fetchAuthSession(contextSpec)
        return !!session.tokens
      } catch (error) {
        console.error(error);
        return false;
      }
    },
  });
  return isAuthenticated;
};

export const getIsUserAuthorizedFromServer = async (group: AppUserGroup) => {
  const isAuthorized = await runWithAmplifyServerContext({
    nextServerContext: { cookies },
    operation: async (contextSpec: AmplifyServer.ContextSpec) => {
      try {
        const session = await fetchAuthSession(contextSpec)
        const groups = INTERNAL__getSessionGroups(session)
        return !!session.tokens && !!groups?.find((g) => g === group)
      } catch (error) {
        console.error(error);
        return false;
      }
    },
  });
  return isAuthorized;
};


export const signOutFromServer = async () => {
  const currentUser = runWithAmplifyServerContext({
    nextServerContext: { cookies },
    operation: () => signOut(),
  });
  return currentUser;
};

const INTERNAL__sessionGroupKey = 'cognito:groups'

function INTERNAL__getSessionGroups(session: AuthSession): AppUserGroup[] | undefined {
  const groups = session.tokens?.accessToken.payload[INTERNAL__sessionGroupKey]
  if (!groups) {
    return
  }

  if (typeof groups === 'object') {
    return groups as AppUserGroup[]
  }

  if (typeof groups === 'string') {
    return JSON.parse(groups) as AppUserGroup[]
  }
}
